'use client';
import React from 'react';
import Link from 'next/link';
import { usePathname } from 'next/navigation';
import { LayoutDashboard, FileText, Users, Calendar, Image, LogOut, UserCircle, Globe, Sparkles } from 'lucide-react';

export default function AdminLayout({ children }: { children: React.ReactNode }) {
  const pathname = usePathname();
  const isLoginPage = pathname === '/admin/login' || pathname === '/admin/forgot-password';

  if (isLoginPage) {
    return (
      <div className="min-h-screen bg-slate-100 flex items-center justify-center p-4">
        {children}
      </div>
    );
  }

  const NavItem = ({ href, icon: Icon, label }: { href: string, icon: any, label: string }) => {
    // Exact match for dashboard, startsWith for others. Safe navigation using optional chaining.
    const isActive = href === '/admin' ? pathname === href : pathname?.startsWith(href);
    
    return (
      <Link 
        href={href} 
        className={`group relative flex items-center gap-2.5 px-4 py-2.5 rounded-xl transition-all duration-200 text-[14px] ${
          isActive 
            ? 'bg-gradient-to-r from-white/20 to-white/10 text-white font-semibold shadow-[inset_0_0_0_1px_rgba(255,255,255,0.14)] backdrop-blur' 
            : 'hover:bg-white/8 text-slate-300 hover:text-white'
        }`}
      >
        {isActive && <span className="absolute left-0 top-2 bottom-2 w-1 rounded-full bg-gradient-to-b from-[#16A34A] to-[#BE123C]"></span>}
        <span className={`w-8 h-8 rounded-lg flex items-center justify-center transition ${isActive ? 'bg-white/15 text-white' : 'bg-white/5 text-slate-300 group-hover:bg-white/10'}`}>
          <Icon size={16} className={isActive ? 'text-white' : ''} />
        </span>
        {label}
      </Link>
    );
  };

  return (
    <div className="min-h-screen bg-[linear-gradient(180deg,#f3f6fb_0%,#e9eef6_100%)] flex">
      {/* Sidebar */}
      <aside className="w-72 bg-[linear-gradient(180deg,#020617_0%,#08153a_100%)] text-white flex flex-col fixed h-full z-10 border-r border-slate-800/80 shadow-[10px_0_40px_-28px_rgba(2,6,23,0.9)]">
        <div className="p-6 border-b border-white/10 bg-white/[0.02] flex items-center gap-3">
          <div className="bg-white p-1.5 rounded-md shadow-sm">
             <img src="/logo.png" alt="Logo" className="h-6 w-auto" />
          </div>
          <div>
            <h1 className="text-lg font-bold font-display tracking-tight text-white">Admin</h1>
            <p className="text-[10px] text-slate-400">UKBCCI Control Panel</p>
          </div>
        </div>
        <div className="px-4 pt-4">
          <div className="rounded-2xl border border-white/10 bg-white/[0.04] px-4 py-3">
            <p className="text-[10px] uppercase tracking-[0.2em] text-slate-400 font-semibold">Workspace</p>
            <p className="text-[13px] text-white font-semibold mt-1 flex items-center gap-2">
              <Sparkles size={14} className="text-primary" />
              Content Management
            </p>
          </div>
        </div>
        <nav className="flex-1 p-4 space-y-1 overflow-y-auto">
          <NavItem href="/admin" icon={LayoutDashboard} label="Dashboard" />
          
          <div className="pt-4 pb-2 px-4 text-[10px] font-bold text-slate-500 uppercase tracking-[0.2em]">Content</div>
          <NavItem href="/admin/posts" icon={FileText} label="Posts" />
          <NavItem href="/admin/events" icon={Calendar} label="Events" />
          <NavItem href="/admin/gallery" icon={Image} label="Gallery" />
          <NavItem href="/admin/leadership" icon={Users} label="Leadership" />
          
          <div className="pt-4 pb-2 px-4 text-[10px] font-bold text-slate-500 uppercase tracking-[0.2em]">System</div>
          <NavItem href="/admin/profile" icon={UserCircle} label="My Profile" />
        </nav>
        <div className="p-4 border-t border-white/10 bg-white/[0.02]">
          <Link href="/" target="_blank" className="flex items-center gap-3 px-4 py-2.5 rounded-xl hover:bg-white/10 text-slate-300 hover:text-white transition mb-2 text-[14px]">
            <Globe size={18} /> Visit Live Site
          </Link>
          <Link href="/api/auth/logout" className="flex items-center gap-3 px-4 py-2.5 rounded-xl hover:bg-red-500/15 text-red-300 hover:text-red-100 transition text-[14px]">
            <LogOut size={18} /> Logout
          </Link>
        </div>
      </aside>

      {/* Main Content */}
      <main className="flex-1 ml-72 p-8 md:p-10">
        <div className="max-w-6xl mx-auto">
          {children}
        </div>
      </main>
    </div>
  );
}
