import React from 'react';
import Navbar from '@/components/Navbar';
import Footer from '@/components/Footer';
import dbConnect from '@/lib/db';
import GalleryItem from '@/models/GalleryItem';
import Link from 'next/link';
import { ArrowLeft, ArrowRight } from 'lucide-react';
import GalleryAlbums from '@/components/GalleryAlbums';

export const revalidate = 60;

async function getGallery(page: number = 1) {
  try {
    await dbConnect();
    const limit = 20;
    const skip = (page - 1) * limit;

    const items = await (GalleryItem as any).find({ status: 'published' })
      .sort({ order: 1 }) // Sorted by order
      .skip(skip)
      .limit(limit)
      .lean();

    const total = await (GalleryItem as any).countDocuments({ status: 'published' });

    return {
      items: JSON.parse(JSON.stringify(items)),
      totalPages: Math.ceil(total / limit),
      currentPage: page
    };
  } catch (e) {
    return { items: [], totalPages: 0, currentPage: 1 };
  }
}

export default async function GalleryPage(props: { searchParams: Promise<{ page?: string }> }) {
  const searchParams = await props.searchParams;
  const page = Number(searchParams.page) || 1;
  const { items, totalPages, currentPage } = await getGallery(page);

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />
      
      <main className="flex-grow bg-white dark:bg-slate-900">
        <div className="relative bg-slate-950 py-20 text-center overflow-hidden">
             <div className="absolute inset-0 bg-[radial-gradient(circle_at_top_right,rgba(22,163,74,0.18),transparent_45%),radial-gradient(circle_at_top_left,rgba(14,58,138,0.25),transparent_40%)]"></div>
             <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-[#0E3A8A] via-[#16A34A] to-[#BE123C]"></div>
             <div className="relative max-w-4xl mx-auto px-4">
               <p className="inline-flex px-4 py-1.5 rounded-full border border-white/20 text-xs font-semibold uppercase tracking-[0.2em] text-slate-300 mb-5">
                 Featured Collections
               </p>
               <h1 className="text-4xl md:text-6xl font-bold text-white font-display mb-4">Gallery</h1>
               <p className="text-slate-300 text-base md:text-lg">Capturing moments of business, trade and community connections.</p>
             </div>
        </div>

        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
           {items.length === 0 ? (
             <p className="text-center text-slate-500 text-base">No images found.</p>
           ) : (
             <GalleryAlbums items={items} />
           )}

           {/* Pagination */}
           {totalPages > 1 && (
            <div className="flex justify-center mt-16 gap-3">
              {currentPage > 1 && (
                <Link href={`/gallery?page=${currentPage - 1}`} className="flex items-center gap-2 px-4 py-2 bg-white border border-slate-200 rounded-lg shadow-sm hover:bg-slate-100 text-sm font-medium transition">
                  <ArrowLeft size={16} /> Prev
                </Link>
              )}
              <span className="px-4 py-2 text-sm text-slate-600 bg-slate-50 border border-slate-200 rounded-lg">Page {currentPage} of {totalPages}</span>
              {currentPage < totalPages && (
                <Link href={`/gallery?page=${currentPage + 1}`} className="flex items-center gap-2 px-4 py-2 bg-white border border-slate-200 rounded-lg shadow-sm hover:bg-slate-100 text-sm font-medium transition">
                  Next <ArrowRight size={16} />
                </Link>
              )}
            </div>
          )}
        </div>
      </main>

      <Footer />
    </div>
  );
}
