import React from 'react';
import { ArrowRight, Calendar, Clock, MapPin } from 'lucide-react';
import Link from 'next/link';

interface Event {
  _id?: string;
  slug?: string;
  title: string;
  subtitle?: string;
  date: string; // Formatted date string or Date object
  image: string;
  location?: string;
  time?: string;
  startDate?: string;
  posterImage?: string;
  locationText?: string;
  timeText?: string;
}

const Events = ({ upcoming, past }: { upcoming: Event | null, past: Event[] }) => {
  const formatDate = (dateStr?: string) => {
    if (!dateStr) return '';
    const d = new Date(dateStr);
    return d.toLocaleDateString('en-GB', { day: 'numeric', month: 'long', year: 'numeric', weekday: 'long' });
  };

  return (
    <>
      {/* Upcoming Event Section */}
      {upcoming && (
        <section id="events" className="relative py-24 bg-[linear-gradient(180deg,#030d26_0%,#0b1d45_100%)] overflow-hidden border-t border-slate-800">
          <div className="absolute inset-0 bg-[radial-gradient(circle_at_10%_20%,rgba(22,163,74,0.2),transparent_30%),radial-gradient(circle_at_90%_80%,rgba(190,18,60,0.18),transparent_36%)]"></div>
          <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-[#0E3A8A] via-[#16A34A] to-[#BE123C]"></div>
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
            <div className="flex items-center gap-2 mb-3">
              <div className="w-10 h-0.5 bg-primary"></div>
              <span className="text-sm font-bold uppercase tracking-widest text-primary">Don't Miss</span>
              <div className="w-2 h-2 rounded-full bg-[#BE123C]"></div>
            </div>
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4 font-display">
              Upcoming <span className="text-primary">Event</span>
            </h2>
            <p className="text-slate-300 mb-10 max-w-2xl">
              Discover our next major networking opportunity and reserve your place.
            </p>

            <div className="grid lg:grid-cols-12 gap-6">
              <div className="lg:col-span-7 relative rounded-3xl overflow-hidden border border-white/15 min-h-[470px] shadow-[0_28px_56px_-30px_rgba(2,6,23,0.85)] group">
                <img
                  src={upcoming.posterImage || upcoming.image}
                  alt={upcoming.title}
                  className="w-full h-full object-cover transition duration-700 group-hover:scale-105"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-slate-950/85 via-slate-900/35 to-transparent"></div>
                <div className="absolute top-5 left-5 inline-flex items-center gap-2 rounded-full bg-emerald-500/90 px-4 py-2 text-xs font-bold uppercase tracking-wider text-white shadow-lg">
                  <span className="w-2 h-2 rounded-full bg-white animate-pulse"></span>
                  Upcoming
                </div>
                <div className="absolute bottom-6 left-6 right-6">
                  <h3 className="text-3xl md:text-4xl font-bold text-white leading-tight mb-2">{upcoming.title}</h3>
                  {upcoming.subtitle && <p className="text-emerald-200 font-semibold">{upcoming.subtitle}</p>}
                </div>
              </div>

              <div className="lg:col-span-5 rounded-3xl border border-white/15 bg-white/10 backdrop-blur-xl p-6 md:p-8 shadow-[0_28px_56px_-30px_rgba(2,6,23,0.85)]">
                <div className="inline-flex items-center rounded-full border border-white/20 bg-white/10 px-4 py-1.5 text-[11px] font-bold uppercase tracking-[0.18em] text-white/85 mb-5">
                  Event Information
                </div>

                <div className="space-y-3 mb-8">
                  <div className="flex gap-3 rounded-2xl border border-white/15 bg-white/10 px-4 py-3">
                    <div className="w-10 h-10 rounded-xl bg-white/10 border border-white/20 flex items-center justify-center text-emerald-300 flex-shrink-0">
                      <Calendar size={18} />
                    </div>
                    <div>
                      <p className="text-[11px] uppercase tracking-widest text-slate-300 font-semibold">Date</p>
                      <p className="text-white font-semibold">{formatDate(upcoming.startDate)}</p>
                    </div>
                  </div>

                  {(upcoming.timeText || upcoming.time) && (
                    <div className="flex gap-3 rounded-2xl border border-white/15 bg-white/10 px-4 py-3">
                      <div className="w-10 h-10 rounded-xl bg-white/10 border border-white/20 flex items-center justify-center text-emerald-300 flex-shrink-0">
                        <Clock size={18} />
                      </div>
                      <div>
                        <p className="text-[11px] uppercase tracking-widest text-slate-300 font-semibold">Time</p>
                        <p className="text-white font-semibold">{upcoming.timeText || upcoming.time}</p>
                      </div>
                    </div>
                  )}

                  {(upcoming.locationText || upcoming.location) && (
                    <div className="flex gap-3 rounded-2xl border border-white/15 bg-white/10 px-4 py-3">
                      <div className="w-10 h-10 rounded-xl bg-white/10 border border-white/20 flex items-center justify-center text-emerald-300 flex-shrink-0">
                        <MapPin size={18} />
                      </div>
                      <div>
                        <p className="text-[11px] uppercase tracking-widest text-slate-300 font-semibold">Location</p>
                        <p className="text-white font-semibold">{upcoming.locationText || upcoming.location}</p>
                      </div>
                    </div>
                  )}
                </div>

                <Link href={`/event/${upcoming.slug}`} className="inline-flex w-full md:w-auto items-center justify-center gap-2 rounded-full bg-white text-slate-900 px-7 py-3.5 text-sm font-bold uppercase tracking-wider transition hover:bg-primary hover:text-white shadow-lg">
                  View Details <ArrowRight size={18} />
                </Link>
              </div>
            </div>
          </div>
        </section>
      )}

      {/* Past Events Section */}
      <section className="relative py-24 bg-[linear-gradient(180deg,#020617_0%,#08122d_100%)] border-t border-slate-800 overflow-hidden">
        <div className="absolute inset-0 bg-[radial-gradient(circle_at_top_left,rgba(14,58,138,0.25),transparent_35%),radial-gradient(circle_at_bottom_right,rgba(190,18,60,0.16),transparent_40%)]"></div>
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-end mb-12">
            <div>
              <div className="flex items-center gap-2 mb-4">
                <div className="w-10 h-0.5 bg-primary"></div>
                <span className="text-sm font-bold uppercase tracking-widest text-primary">Highlights</span>
                <div className="w-2 h-2 rounded-full bg-[#BE123C]"></div>
              </div>
              <h2 className="text-4xl md:text-5xl font-bold text-white font-display">
                Past <span className="text-primary">Events</span>
              </h2>
              <p className="text-slate-300 mt-3 max-w-2xl">
                Key moments from UKBCCI networking events, trade delegations, and regional business gatherings.
              </p>
            </div>
            <Link href="/events" className="hidden sm:flex items-center gap-2 text-slate-200 hover:text-white transition text-base border border-slate-600 rounded-full px-6 py-2.5 hover:bg-slate-800/80">
              View All <ArrowRight size={16} />
            </Link>
          </div>

          <div className="relative z-10 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
            {past && past.map((event, idx) => (
              <Link href={`/event/${event.slug}`} key={idx} className="group cursor-pointer rounded-2xl border border-slate-700/70 bg-slate-900/65 backdrop-blur-sm overflow-hidden hover:border-slate-500 transition-all duration-300 hover:-translate-y-1.5 shadow-[0_24px_42px_-28px_rgba(2,6,23,0.95)]">
                <div className="relative">
                  <div className="h-1.5 bg-gradient-to-r from-[#0E3A8A] via-[#16A34A] to-[#BE123C]"></div>
                  <div className="relative h-52 overflow-hidden">
                    <img src={event.posterImage || event.image} alt={event.title} className="w-full h-full object-cover group-hover:scale-110 transition duration-700 opacity-90 group-hover:opacity-100" />
                    <div className="absolute inset-0 bg-gradient-to-t from-slate-950/80 via-slate-900/25 to-transparent"></div>
                    <div className="absolute top-3 left-3 bg-emerald-500/85 backdrop-blur text-white text-[11px] font-semibold px-3 py-1.5 rounded-full border border-white/20">
                      Past Event
                    </div>
                    <div className="absolute bottom-3 left-3 bg-black/45 backdrop-blur text-white text-[11px] font-semibold px-3 py-1.5 rounded-full border border-white/15">
                      {new Date(event.startDate || event.date).toLocaleDateString()}
                    </div>
                  </div>
                </div>
                <div className="p-4 bg-[linear-gradient(180deg,rgba(15,23,42,0.2)_0%,rgba(15,23,42,0.45)_100%)]">
                  <h3 className="text-white font-bold text-base group-hover:text-primary transition line-clamp-2 leading-snug">{event.title}</h3>
                  <span className="inline-flex items-center gap-2 mt-3 text-xs text-slate-300 group-hover:text-white transition">
                    View Gallery <ArrowRight size={13} className="group-hover:translate-x-1 transition-transform" />
                  </span>
                </div>
              </Link>
            ))}
          </div>
        </div>
      </section>
    </>
  );
};

export default Events;
