import React from 'react';
import { ArrowRight } from 'lucide-react';
import Link from 'next/link';

interface Leader {
  name: string;
  role: string;
  image: string;
  roleColor: string;
}

const Leadership = ({ leaders }: { leaders: Leader[] }) => {
  // Use passed props, or fallback to empty array if data fetching fails
  const displayLeaders = leaders || [];

  return (
    <section id="leadership" className="relative py-24 bg-slate-950 text-white overflow-hidden">
      <div className="absolute inset-0 bg-[radial-gradient(circle_at_top_left,rgba(14,58,138,0.2),transparent_40%),radial-gradient(circle_at_bottom_right,rgba(190,18,60,0.18),transparent_45%)]"></div>
      <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-[#0E3A8A] via-[#16A34A] to-[#BE123C]"></div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
        <div className="flex flex-col md:flex-row justify-between items-end mb-16 gap-6">
          <div>
            <div className="flex items-center gap-2 mb-4">
              <div className="w-10 h-0.5 bg-primary"></div>
              <span className="text-sm font-bold uppercase tracking-widest text-primary">Leadership</span>
              <div className="w-2 h-2 rounded-full bg-[#BE123C]"></div>
            </div>
            <h2 className="text-4xl md:text-5xl font-bold font-display">
              Meet Our <span className="text-primary">Visionary Leaders</span>
            </h2>
          </div>
          <p className="text-slate-400 text-base max-w-sm md:text-right leading-relaxed">
            Providing strategic vision to strengthen UK-BD trade relations and empower our community.
          </p>
        </div>

        <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-6">
          {displayLeaders.map((leader, index) => (
            <div key={index} className="group relative">
              <div className="aspect-[3/4] rounded-2xl overflow-hidden bg-slate-900/60 relative shadow-[0_24px_40px_-30px_rgba(0,0,0,0.9)] border border-white/10">
                <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-[#0E3A8A] via-[#16A34A] to-[#BE123C] z-10"></div>
                <img
                  src={leader.image || '/placeholder-leader.jpg'}
                  alt={leader.name}
                  loading="lazy"
                  className="w-full h-full object-cover group-hover:scale-105 transition duration-500 filter grayscale group-hover:grayscale-0"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-slate-900 via-transparent to-transparent opacity-90"></div>
                <div className="absolute bottom-0 left-0 w-full p-4">
                  <span className={`inline-block px-2.5 py-1 rounded-full ${leader.roleColor || 'bg-primary'} text-[10px] font-bold text-white mb-2 uppercase tracking-wide`}>
                    {leader.role}
                  </span>
                  <h3 className="text-sm font-bold leading-tight">{leader.name}</h3>
                </div>
              </div>
            </div>
          ))}
        </div>

        <div className="mt-16 text-center">
          <Link href="/leadership" className="inline-flex items-center justify-center bg-white text-slate-900 px-8 py-4 rounded-full text-base font-bold hover:bg-primary hover:text-white transition shadow-lg shadow-white/10">
            View Full Team <ArrowRight size={18} className="ml-2" />
          </Link>
        </div>
      </div>
    </section>
  );
};

export default Leadership;
