'use client';
import React, { useState, useEffect } from 'react';
import Link from 'next/link';
import { usePathname } from 'next/navigation';
import {
  Phone, Mail, MapPin, Facebook, Linkedin, Instagram, Search,
  ArrowRight, Menu, X, ChevronDown, Heart, Trophy
} from 'lucide-react';

const Navbar: React.FC = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);
  const pathname = usePathname();

  // Handle scroll effect
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 20);
    };
    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const isActive = (path: string) => pathname === path;
  const navLinkClass = (path: string) =>
    `group relative px-3.5 py-2 text-[13px] font-semibold rounded-xl transition-all duration-200 ${
      isActive(path)
        ? 'text-slate-900 bg-gradient-to-b from-emerald-100/70 to-emerald-50 border border-emerald-200/70 shadow-[inset_0_1px_0_rgba(255,255,255,0.9)]'
        : 'text-slate-600 hover:text-slate-900 hover:bg-slate-100/80'
    }`;

  return (
    <>
      {/* Top Bar - Sleek & Dark */}
      <div className="bg-[linear-gradient(90deg,#0b1120_0%,#0a1328_100%)] text-slate-400 text-[11px] font-medium tracking-wide py-1.5 px-4 border-b border-white/5 hidden md:block relative z-50">
        <div className="max-w-7xl mx-auto flex justify-between items-center h-8">
          <div className="flex space-x-6">
            <a href="tel:07734394192" className="flex items-center gap-1.5 hover:text-white transition-colors duration-200 group">
              <Phone size={13} className="text-primary group-hover:text-emerald-400" />
              <span>07734 394192</span>
            </a>
            <a href="mailto:info@ukbcci.org.uk" className="flex items-center gap-1.5 hover:text-white transition-colors duration-200 group">
              <Mail size={13} className="text-primary group-hover:text-emerald-400" />
              <span>info@ukbcci.org.uk</span>
            </a>
            <span className="flex items-center gap-1.5 cursor-default group">
              <MapPin size={13} className="text-primary group-hover:text-emerald-400" />
              <span>London, United Kingdom</span>
            </span>
          </div>
          <div className="flex items-center gap-5">
            <div className="flex space-x-4 items-center border-r border-white/10 pr-5 h-full">
              <a href="#" className="hover:text-primary transition-colors transform hover:scale-110 duration-200"><Linkedin size={14} /></a>
              <a href="#" className="hover:text-primary transition-colors transform hover:scale-110 duration-200"><Facebook size={14} /></a>
              <a href="#" className="hover:text-primary transition-colors transform hover:scale-110 duration-200"><Instagram size={14} /></a>
              <button className="hover:text-primary transition-colors"><Search size={14} /></button>
            </div>
            <Link href="/membership" className="flex items-center gap-1 text-emerald-400 hover:text-emerald-300 transition-colors group font-semibold">
              Become a Member <ArrowRight size={13} className="group-hover:translate-x-1 transition-transform" />
            </Link>
          </div>
        </div>
      </div>

      {/* Main Navigation */}
      <nav
        className={`sticky top-0 z-40 w-full border-b transition-all duration-500 ease-in-out ${scrolled
          ? 'bg-white/80 backdrop-blur-md border-slate-200/70 shadow-[0_18px_34px_-26px_rgba(15,23,42,0.35)]'
          : 'bg-white border-slate-200/60'
          }`}
      >
        <div className="max-w-7xl mx-auto px-3 sm:px-4 lg:px-4 py-2">
          <div className={`flex justify-between items-center rounded-2xl border border-slate-200/70 bg-white/95 px-4 lg:px-5 transition-all duration-300 ${scrolled ? 'h-[74px] shadow-[0_18px_30px_-24px_rgba(15,23,42,0.3)]' : 'h-[78px] shadow-[0_16px_24px_-22px_rgba(15,23,42,0.22)]'}`}>

            {/* Logo Area */}
            <div className="flex-shrink-0 flex items-center">
              <Link href="/" className="block relative group">
                <img
                  src="/logo.png"
                  alt="UKBCCI Logo"
                  className={`transition-all duration-500 ${scrolled ? 'h-14' : 'h-[68px]'} w-auto`}
                />
              </Link>
            </div>

            {/* Desktop Menu */}
            <div className="hidden lg:flex items-center gap-1 bg-slate-50/80 border border-slate-200/80 rounded-2xl px-2 py-1.5">
              {[
                { name: 'Home', path: '/' },
                { name: 'About', path: '/about' },
                { name: 'Leadership', path: '/leadership' },
              ].map((item) => (
                <Link
                  key={item.name}
                  href={item.path}
                  className={navLinkClass(item.path)}
                >
                  <span className="relative z-10">{item.name}</span>
                </Link>
              ))}

              {/* Dropdown: Legacy */}
              <div className="relative group px-0.5">
                <button className={`group relative flex items-center gap-1 px-3.5 py-2 text-[13px] font-semibold transition-all duration-200 rounded-xl border ${pathname?.startsWith('/legacy') ? 'text-slate-900 bg-gradient-to-b from-emerald-100/70 to-emerald-50 border-emerald-200/70' : 'text-slate-600 border-transparent hover:bg-slate-100 hover:text-slate-900'}`}>
                  <span className="relative z-10">Legacy</span>
                  <ChevronDown size={13} className="group-hover:rotate-180 transition-transform duration-300" />
                </button>

                {/* Dropdown Menu */}
                <div className="absolute top-full left-1/2 -translate-x-1/2 pt-4 w-80 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-300 transform translate-y-2 group-hover:translate-y-0">
                  <div className="bg-white/95 backdrop-blur rounded-2xl shadow-2xl border border-slate-200 overflow-hidden ring-1 ring-black/5 p-3">
                    <Link href="/legacy/remembrance" className="flex items-start gap-4 p-3 rounded-xl hover:bg-slate-50 transition-colors group/item border border-transparent hover:border-slate-200">
                      <div className="w-10 h-10 rounded-full bg-rose-50 text-rose-500 flex items-center justify-center flex-shrink-0 group-hover/item:bg-rose-500 group-hover/item:text-white transition-all">
                        <Heart size={20} />
                      </div>
                      <div>
                        <span className="block text-sm font-bold text-slate-800">Remembrance</span>
                        <span className="text-xs text-slate-500 leading-tight block mt-1">Honoring our departed leaders and directors</span>
                      </div>
                    </Link>
                    <Link href="/legacy/icons" className="flex items-start gap-4 p-3 rounded-xl hover:bg-slate-50 transition-colors group/item border border-transparent hover:border-slate-200">
                      <div className="w-10 h-10 rounded-full bg-amber-50 text-amber-600 flex items-center justify-center flex-shrink-0 group-hover/item:bg-amber-500 group-hover/item:text-white transition-all">
                        <Trophy size={20} />
                      </div>
                      <div>
                        <span className="block text-sm font-bold text-slate-800">UKBCCI Icons</span>
                        <span className="text-xs text-slate-500 leading-tight block mt-1">Celebrating lifetime achievement award winners</span>
                      </div>
                    </Link>
                  </div>
                </div>
              </div>

              {[
                { name: 'Services', path: '/services' },
                { name: 'Regions', path: '/regions' },
                { name: 'Events', path: '/events' },
                { name: 'News', path: '/category/news' },
                { name: 'Blogs', path: '/category/blogs' },
                { name: 'Gallery', path: '/gallery' },
                { name: 'Contact', path: '/contact' },
              ].map((item) => (
                <Link
                  key={item.name}
                  href={item.path}
                  className={navLinkClass(item.path)}
                >
                  <span className="relative z-10">{item.name}</span>
                </Link>
              ))}
            </div>

            {/* Right Side Actions */}
            <div className="hidden lg:flex items-center gap-2 pl-2">
              <Link
                href="/membership"
                className="group relative overflow-hidden rounded-xl bg-[#0b1120] px-5 py-2.5 text-[11px] font-bold uppercase tracking-[0.08em] text-white shadow-md transition-all hover:bg-[#16A34A] hover:shadow-xl hover:shadow-emerald-500/30 border border-[#0b1120]"
              >
                <span className="relative flex items-center gap-2">
                  Join UKBCCI <ArrowRight size={16} className="transition-transform group-hover:translate-x-1" />
                </span>
              </Link>
            </div>

            {/* Mobile Menu Button */}
            <div className="flex lg:hidden">
              <button
                onClick={() => setIsOpen(!isOpen)}
                className="text-slate-600 hover:text-slate-900 p-2 rounded-full transition-colors focus:outline-none hover:bg-slate-100"
              >
                {isOpen ? <X size={26} /> : <Menu size={26} />}
              </button>
            </div>
          </div>
        </div>

        {/* Mobile Menu Overlay */}
        <div
          className={`lg:hidden fixed inset-0 z-50 bg-slate-900/60 backdrop-blur-sm transition-opacity duration-300 ${isOpen ? 'opacity-100 visible' : 'opacity-0 invisible'}`}
          onClick={() => setIsOpen(false)}
        ></div>

        {/* Mobile Menu Panel */}
        <div className={`lg:hidden fixed top-0 right-0 z-50 w-[85%] max-w-sm h-full bg-white shadow-2xl transition-transform duration-300 transform ${isOpen ? 'translate-x-0' : 'translate-x-full'}`}>
          <div className="flex flex-col h-full">
            <div className="flex items-center justify-between p-6 border-b border-slate-100">
              <div className="w-36">
                <img src="/logo.png" alt="Logo" className="w-full h-auto" />
              </div>
              <button onClick={() => setIsOpen(false)} className="p-2 text-slate-400 hover:text-red-500 transition-colors bg-slate-50 rounded-full">
                <X size={22} />
              </button>
            </div>

            <div className="flex-1 overflow-y-auto py-6 px-4 space-y-1">
              {[
                { name: 'Home', path: '/' },
                { name: 'About', path: '/about' },
                { name: 'Leadership', path: '/leadership' },
                { name: 'Services', path: '/services' },
                { name: 'Regions', path: '/regions' },
                { name: 'Events', path: '/events' },
                { name: 'News', path: '/category/news' },
                { name: 'Blogs', path: '/category/blogs' },
                { name: 'Gallery', path: '/gallery' },
                { name: 'Contact', path: '/contact' },
              ].map((item) => (
                <Link
                  key={item.name}
                  href={item.path}
                  onClick={() => setIsOpen(false)}
                  className={`block px-4 py-3.5 rounded-xl font-medium text-base transition-all ${isActive(item.path)
                    ? 'bg-emerald-50 text-primary translate-x-1'
                    : 'text-slate-700 hover:bg-slate-50 hover:text-primary hover:translate-x-1'
                    }`}
                >
                  {item.name}
                </Link>
              ))}

              <div className="border-t border-slate-100 my-4 pt-4">
                <p className="px-4 text-xs font-bold text-slate-400 uppercase tracking-widest mb-3">Legacy</p>
                <Link href="/legacy/remembrance" onClick={() => setIsOpen(false)} className="flex items-center gap-3 px-4 py-3.5 rounded-xl text-slate-700 hover:bg-slate-50 hover:text-primary transition-colors text-base">
                  <div className="w-8 h-8 rounded-full bg-red-50 text-red-500 flex items-center justify-center"><Heart size={16} /></div>
                  Remembrance
                </Link>
                <Link href="/legacy/icons" onClick={() => setIsOpen(false)} className="flex items-center gap-3 px-4 py-3.5 rounded-xl text-slate-700 hover:bg-slate-50 hover:text-primary transition-colors text-base">
                  <div className="w-8 h-8 rounded-full bg-yellow-50 text-yellow-600 flex items-center justify-center"><Trophy size={16} /></div>
                  UKBCCI Icons
                </Link>
              </div>
            </div>

            <div className="p-6 border-t border-slate-100 bg-slate-50">
              <Link href="/membership" onClick={() => setIsOpen(false)} className="block w-full bg-[#0b1120] text-white text-center py-4 rounded-xl font-bold hover:bg-primary transition-colors shadow-lg shadow-slate-900/10 flex items-center justify-center gap-2 text-lg">
                Join UKBCCI <ArrowRight size={18} />
              </Link>
            </div>
          </div>
        </div>
      </nav>
    </>
  );
};

export default Navbar;
